/*
 * Controlling an LED Matrix Module Samsung SLM1606M/SLM1608M
 * connected to printer port:
 *
 * Printer   Module
 * -----------------
 * D0 (2)    SELECT (CN2-2)
 * D1 (3)    RED    (CN3-2)
 * D2 (4)    GREEN  (CN3-4)
 * D3 (6)    CLOCK  (CN3-6)
 * D4 (8)    BRIGHT (CN3-8)
 * D5 (9)    RESET  (CN3-10)
 * GND (20)  GND    (CN3-3)
 *
 * Written by Thorsten Erdmann 06/2003 (thorsten.erdmann@gmx.de)
 */

/**************************************************************
 **** CAUTION: The module can be destroyed if applied      ****
 ****          static signals, so NEVER interrupt the      ****
 ****          program without switching off module's      ****
 ****          power source first !!!                      ****
 **************************************************************/

#include <stdlib.h>
#include <stdio.h>

#define LPTADR 0x378
#define SELECT 0
#define RED    1
#define GREEN  2
#define CLOCK  3
#define BRIGHT 4
#define RESET  5

int lptstate=0;
unsigned short bufgreen[16], bufred[16], altered=1;

void setbit(short b)
{
  outportb(LPTADR, (lptstate|=(1<<b)));
}

void clrbit(short b)
{
  outportb(LPTADR, (lptstate&=~(1<<b)));
}

void lptInit(void)
{
  outportb(LPTADR+2,0); /* no interrupts, D0..7 output */
  lptstate=0;
  setbit(RESET);        /* reset module      */
  setbit(BRIGHT);       /* switch off module */
}

void wait(void)
{
  int i;
  for (i=0;i<20;i++) outportb(LPTADR,lptstate);
}

void pulse(short bit)
{
  setbit(bit);
  clrbit(bit);
}

/* Clear Module */
void ModuleClear(void)
{
  memset(bufgreen,0,32);
  memset(bufred,0,32);
}

/* Multiplex Module, must be called periodically to avoid module destruction */
void ModuleMux(void)
{
  static short x=0,y=0;
  static short mask=1;

  if (!x && !y)
  {
    pulse(RESET);
    if (altered) setbit(SELECT);
  }
  clrbit(BRIGHT);
  if (altered)
  {
    if (bufgreen[y]&mask)
      setbit(GREEN);
    else
      clrbit(GREEN);
    if (bufred[y]&mask)
      setbit(RED);
    else
      clrbit(RED);
  }

  pulse(CLOCK);
  x++;
  mask<<=1;
  if (x>15)
  {
    x=0;
    mask=1;
    y++;
    if (y>15)
    {
      y=0;
      altered=0;
      clrbit(SELECT);
    }
  }
}

void ModuleSetPixel(short x, short y, short color)
{
  if (color&1)
    bufred[y]|=(1<<x);
  else
    bufred[y]&=~(1<<x);
  if (color&2)
    bufgreen[y]|=(1<<x);
  else
    bufgreen[y]&=~(1<<x);
  altered=1;
}

void ModuleSetHLine(short x, short y, short len, short color)
{
  short i;
  for (i=x; i<x+len; i++) ModuleSetPixel(i,y,color);
}

void ModuleSetVLine(short x, short y, short len, short color)
{
  short i;
  for (i=y; i<y+len; i++) ModuleSetPixel(x,i,color);
}

int main(void)
{
  lptInit();
  ModuleClear();
  ModuleSetPixel(2,2,1);
  ModuleSetPixel(2,4,2);
  ModuleSetHLine(1,1,14,3);
  ModuleSetHLine(1,14,14,3);
  ModuleSetVLine(1,1,14,1);
  ModuleSetVLine(14,1,14,2);
  puts("Press any key to stop");
  do
  {
    ModuleMux();
    wait();
  }
  while (!kbhit());
  getch();
  lptInit();
  exit(0);
}

