/* LED Matrix display driver (Commandline Interpreter)
 *
 * Controlling several LED Matrix Modules Samsung SLM1606M/SLM1608M
 * connected to port B/D of Ethernut:
 *
 * Printer   Module
 * -----------------
 * D0 (2)    SELECT (CN2-2)
 * D1 (3)    RED    (CN3-2)
 * D2 (4)    GREEN  (CN3-4)
 * D3 (6)    CLOCK  (CN3-6)
 * D4 (8)    BRIGHT (CN3-8)
 * D5 (9)    RESET  (CN3-10)
 * GND (20)  GND    (CN3-3)
 *
 * Written by Thorsten Erdmann 06/2003 (thorsten.erdmann@gmx.de)
 */
 
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>
#include <ctype.h>

#include <sys/heap.h>
#include <sys/timer.h>
#include <avr/pgmspace.h>

#include <sys/heap.h>
#include <sys/timer.h>
#include <arpa/inet.h>
#include <netdb.h>

#include "ledmatrix.h"
#include "led.h"
#include "config.h"
#include "ping.h"

#define stricmp strcasecmp

prog_char help_P[]   = 
"\r\nList of commands follows\r\n"
"  PRINT [AT x,y[,w]] [LEFT|CENTERED|RIGHT] \"txt\"\r\n"
"                              draw text\r\n"
"  PIXEL x,y[,color]           draw Pixel\r\n"
"  LINE [x,y] to x,y[,color]   draw line\r\n"
"  HLINE x,y,l[,color]         draw vertical line\r\n"
"  VLINE x,y,l[,color]         draw horizontal line\r\n"
"  RECT x,y,w,h[,color]        draw rectangle\r\n"
"  FRECT x,y,w,h[,color]       draw filled rectangle\r\n"
"  CIRCLE x,y,r[,color]        draw circle\r\n"
"  FCIRCLE x,y,r[,color]       draw filled circle\r\n"
"  TEXTCOLOR color             set text color\r\n"
"  LINECOLOR color             set line color\r\n"
"  COLOR color                 set text and line color\r\n"
"  FONT BIG|SMALL              select font\r\n"
"  CLS                         clear display\r\n" 
"  DRAWDISPLAY [AUTO|MANU]     actualize display or (re)sets autoupdate mode\r\n"
"  INFO                        show configuration info\r\n"
"  POWER ON|OFF                switch on/off the display panel\r\n"
"  TEST                        display a test screen\r\n"
"  SET variable,value          set config data, type set ? for help\r\n"
"  SAVE                        save configuration settings\r\n"
"  DIAG                        hardware diagnose (WARNING: disconnect display!)\r\n"
"  PING hostname|IP-address    ping a host\r\n"
"  NSLOOKUP hostname           lookup a hosts IP address\r\n"
"  QUIT                        end telnet session\r\n"
"  HELP|?                      display this help\r\n";

  prog_char diag_warning_P[]= "\r\nWarning!!!\r\n"
                              "This diagnostic can damage your LED modules\r\n"
                              "So disconnect them before use\r\n\r\n"
                              "Do you want to continue? (y/n)";
  prog_char diag_prompt_P[] = "\r\nPress +/- to change diag mode or ESC to quit\r\n";
	prog_char diag_portd_P[]  = "Pulsing Control Pin, PortD Pin %i\r\n";
	prog_char diag_portb_P[]  = "Pulsing Color Pin,   PortB Pin %i\r\n";
	prog_char diag_msel_P[]   = "Pulsing Module Select %i\r\n";



int txtx=0, txty=0, txtformat=TE_LEFT;
int linex=0, liney=0; 
u_char linecolor=1, txtcolor=1;
char alwaysdraw=1;

char *strpbrk(const char *s1, const char *s2)
{
  const char *scanp;
  int c, sc;

  while ((c = *s1++) != 0)
  {
    for (scanp = s2; (sc = *scanp++) != 0;)
      if (sc == c)
	      return (char *)s1-1;
  }
  return 0;
}

size_t strspn(const char *s1, const char *s2)
{
  const char *p = s1, *spanp;
  char c, sc;

 cont:
  c = *p++;
  for (spanp = s2; (sc = *spanp++) != 0;)
    if (sc == c)
      goto cont;
  return (p - 1 - s1);
}

/*
 * check if char is a space or a tab
 * The C standard routine get confused with special characters
 */
int myisspace(char c)
{
  return ((c==' ') || (c==8) || (c=='\n'));
}

/* Remove trailing spaces from string */
void strrtrim(char *s)
{
  int i;
  if (s==NULL) return;
  i=strlen(s)-1;
  while ((i>=0) && (myisspace(s[i])))
    s[i--]=0;
}

/* Remove leading spaces from string */
void strltrim(char *s)
{
  u_int i=0;
  if (s==NULL) return;
  while ((i<strlen(s)) && (myisspace(s[i]))) i++;
  strcpy(s,&s[i]);
}

/* Remove Comments */
void removeComments(char *s)
{
  u_int i=0, str=0;
  if (s==NULL) return;
  while ((i<strlen(s)) && (s[i]))
  {
    if (s[i]=='"')
      str=~str;
    if ((!str) && ((s[i]=='#') || (s[i]=='\'')))
      s[i]=0;
    else
      i++;
  }
}

/* removes comments, leading and trailing spaces from s */
int trimline(char *s)
{
  if (s==NULL) return 0;
  removeComments(s);
  strrtrim(s);
  strltrim(s);
  return(s[0]!=0);
}

/*
 * like strtok but skips string constants embedded in ""
 */
char *strtoken(char *s)
{
  const char b[]=" \n\t,";
  static char *p=NULL,*p2=NULL;
  static char c;
  if (!s)
  {
   // if (p) *p=c;
    s=p2;
  }
  if (!(*s)) return NULL;
  if (*s=='"')
  {
    p=s+1;
    for (;;)
    {
      if (!(p=strchr(p,'"'))) return NULL;
      if (*(p+1)!='"')
      {
        p++;
        if (*p)
        {
          if ((p=strpbrk(p,b)))
          {
            p2=&p[strspn(p,b)];
            c=*p;
            if (c=='\n') c=0;
            *p=0;
          }
        }
        return s;
      }
      else
        p+=2;
    }
  }
  else
  {
    if ((p=strpbrk(s,b)))
    {
      p2=&p[strspn(p,b)];
      c=*p;
      if (c=='\n') c=0;
      *p=0;
    }
    else
    {
    	p2=NULL;
      return s;
    }
  }
  return s;
}

/*
 * get a string from a quoted string
 */
char* getString(char *s)
{
  char *p;
  if ((s==NULL) || (*s!='"')) return NULL;
  s++;
  p=s;
  for (;;)
  {
    if (!(p=strchr(p,'"'))) return NULL;
    if (*(p+1)!='"')
    {
      *p=0;
      return s;
    }
    else
    {
      p++;
      strcpy(p,p+1);
    }
  }
  return NULL;
}

/*
 * draw a text string
 */
int cmdPrint(FILE *stream)
{
  char *par,*t;
  int x,y,w,f,nl;
  u_char c;
  
  nl=1;
  x=txtx;
  y=txty;
  w=0;
  f=-1;
  c=txtcolor;
  t=NULL;

  par=strtoken(NULL);
  if (!stricmp(par,"AT"))
  {
    par=strtoken(NULL);
    if (!par) return 3;
    x=atoi(par);
    par=strtoken(NULL);
    if (!par) return 3;
    y=atoi(par);
    par=strtoken(NULL);
  }
  if (par)
  {
    if(isdigit(*par))
    {
      w=atoi(par);
      f=txtformat;
      par=strtoken(NULL);
    }
  }
  if (par)
  {
    if (!stricmp(par,"LEFT"))
    {
      f=TE_LEFT;
      par=strtoken(NULL);
    }
    else
      if (!stricmp(par,"CENTERED"))
      {
        f=TE_CENTER;
        par=strtoken(NULL);
      }
      else
        if (!stricmp(par,"RIGHT"))
        {
          f=TE_RIGHT;
          par=strtoken(NULL);
        }
    if (par)
    {
      if (*par=='"')
      {
        if ((*(par+strlen(par)-1))==';') nl=0;
        t=getString(par);
      }
      else return 4;
    }
  }

  if (t)
  {
    if (f<0)
      LEDStr(x,y,c,t);
    else
      LEDFStr(x,y,w,f,c,t);
  }
  if (nl)
  {
    txtx=0;
    txty+=LEDGetStrHeight();
  }
  else
  {
    txtx+=LEDGetStrLength(t);
    if (txtx>MAXX)
    {
      txtx=0;
      txty+=LEDGetStrHeight();
    }
  }
  return 0;
}

/*
 * draw a pixel
 */
int cmdPixel(FILE *stream)
{
  int x,y;
  u_char c;
  char *par;

  c=linecolor;
  x=linex;
  y=liney;

  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (par) 
  {
  	 printf("par:<%s>",par); 
  	 c=(u_char)atoi(par);
  }
  LEDPixel(x,y,c);
  return 0;
}

/*
 * draw a line
 */
int cmdLine(FILE *stream)
{
  int x1,y1,x2,y2;
  u_char c;
  char *par;

  c=linecolor;
  x1=linex;
  y1=liney;

  par=strtoken(NULL);
  if (stricmp(par,"TO"))
  {
    if (!par) return 3;
    x1=atoi(par);
    par=strtoken(NULL);
    if (!par) return 3;
    y1=atoi(par);
    par=strtoken(NULL);
  }
  if ((par) && (!stricmp(par,"TO")))
  {
    par=strtoken(NULL);
    if (!par) return 3;
    x2=atoi(par);
    par=strtoken(NULL);
    if (!par) return 3;
    y2=atoi(par);
    par=strtoken(NULL);
  }
  else
  {
    x2=x1;
    y2=y1;
    x1=linex;
    y1=liney;
  }
  if (par) c=atoi(par);
  linex=x2;
  liney=y2;

  if (x1==x2)
    LEDVLine(x1,y1,y2-y1,c);
  else
    if (y1==y2)
      LEDHLine(x1,y1,x2-x1,c);
    else
      LEDLine(x1,y1,x2,y2,c);
  return 0;
}

/*
 * draw a horizontal line
 */
int cmdHLine(FILE *stream)
{
  int x,y,l;
  u_char c;
  char *par;

  c=linecolor;

  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  l=atoi(par);
  par=strtoken(NULL);
  if (par) c=atoi(par);

  LEDHLine(x,y,l,c);
  return 0;
}

/*
 * draw a vertical line
 */
int cmdVLine(FILE *stream)
{
  int x,y,l;
  u_char c;
  char *par;

  c=linecolor;

  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  l=atoi(par);
  par=strtoken(NULL);
  if (par) c=atoi(par);

  LEDVLine(x,y,l,c);
  return 0;
}

/*
 * draw a circle
 */
int cmdCircle(FILE *stream)
{
  int x,y,r;
  u_char c;
  char *par;

  c=linecolor;
  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  r=atoi(par);
  par=strtoken(NULL);
  if (par)
    c=atoi(par);
  LEDCircle(x,y,r,c);
  return 0;
}

/*
 * draw a filled circle
 */
int cmdFCircle(FILE *stream)
{
  int x,y,r;
  u_char c;
  char *par;

  c=linecolor;
  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  r=atoi(par);
  par=strtoken(NULL);
  if (par)
    c=atoi(par);
  LEDFCircle(x,y,r,c);
  return 0;
}

/*
 * draw a rectangle
 */
int cmdRect(FILE *stream)
{
  int x,y,w,h;
  u_char c;
  char *par;

  c=linecolor;
  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  w=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  h=atoi(par);
  par=strtoken(NULL);
  if (par)
    c=atoi(par);
  LEDRectangle(x,y,w,h,c);
  return 0;
}

/*
 * Draw a filled rectangle
 */
int cmdFRect(FILE *stream)
{
  int x,y,w,h;
  u_char c;
  char *par;

  c=linecolor;
  par=strtoken(NULL);
  if (!par) return 3;
  x=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  y=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  w=atoi(par);
  par=strtoken(NULL);
  if (!par) return 3;
  h=atoi(par);
  par=strtoken(NULL);
  if (par)
    c=atoi(par);
  LEDFRectangle(x,y,w,h,c);
  return 0;
}

/*
 * sets text color
 */
int cmdTextcolor(FILE *stream)
{
  char *par;

  par=strtoken(NULL);
  if (!par) return 3;
  txtcolor=(u_char)atoi(par);
  return 0;
}

/*
 * sets line color
 */
int cmdLinecolor(FILE *stream)
{
  char *par;

  par=strtoken(NULL);
  if (!par) return 3;
  linecolor=(u_char)atoi(par);
  return 0;
}

/*
 * sets line AND textcolor
 */
int cmdColor(FILE *stream)
{
  char *par;

  par=strtoken(NULL);
  if (!par) return 3;
  txtcolor=linecolor=(u_char)atoi(par);
  return 0;
}

/*
 * loads and activates a display font
 */
int cmdFont(FILE *stream)
{
	/*
  char *par;

  par=strtoken(NULL);
  if (!par) return 3;
  if (!stricmp(par,"SMALL"))
    LEDSelectFont(smallfont);
  else
    LEDSelectFont(bigfont);
*/    
  return 0;
}

/*
 * clear display
 */
int cmdCls(FILE *stream)
{
  LEDClear();
  txtcolor=1;
  linecolor=1;
  txtx=txty=linex=liney=0;
  txtformat=TE_LEFT;
  return 0;
}

/*
 * actualize display
 */
int cmdDraw(FILE *stream)
{
	char *par;
	
	par=strtoken(NULL);
	if (!stricmp(par,"AUTO"))
		alwaysdraw=1;
	else
	  if (!stricmp(par,"MANU"))
	    alwaysdraw=0;
	  else
      LEDDraw();
  return 0;
}

/*
 * display configuration
 */
int cmdInfo(FILE *stream)
{
	showConfigInfo(stream);
  return 0;
}

/*
 * set configuration
 */
int cmdSet(FILE *stream)
{
	char *varname, *value;
	varname=strtoken(NULL);
	if (varname) 
		value=strtoken(NULL);
	else
	  value=NULL;	
	return set_config(varname,value,stream);}

/*
 * set configuration
 */
int cmdSave(FILE *stream)
{
  save_config(stream);
  return 0;
}

/*
 * display help 
 */
int cmdHelp(FILE *stream)
{
	fputs_P(help_P,stream);
  return 0;
}

/*
 * quit
 */
int cmdQuit(FILE *stream)
{
  return 1;
}

/*
 * power
 */
int cmdPower(FILE *stream)
{
  char *par;

  par=strtoken(NULL);
  if (!par) return 3;
  if (!strcasecmp_P(par,PSTR("ON")))  LEDPowerOn();	
  if (!strcasecmp_P(par,PSTR("OFF"))) LEDPowerOff();
  
  return 2;
}

/*
 * Test picture
 */
int cmdTest(FILE *stream)
{
	int i;
	LEDClear();
	LEDRectangle(0,0,128,64,1);
	LEDRectangle(2,2,124,60,2);
	LEDRectangle(4,4,120,56,3);
	for (i=0;i<10;i++)
		LEDCircle(22+i*5,32,i*2,i&3);
  return 0;
}
  
/*
 * Hardware diagnose
 */
int cmdDiag(FILE *stream)
{
	char ch;
	fputs_P(diag_warning_P,stream);
	do
	{
		ch=fgetc(stream);
	} while ((ch!='j') && (ch!='y') && (ch!='n'));
	if (ch!='n')
	{
		fputs_P(diag_prompt_P,stream);
		testmode=1;
		do
		{
			if ((testmode>=1) && (testmode<=8))
 	      fprintf_P(stream,diag_portd_P,testmode-1); 
 	    else  
  			if ((testmode>=9) && (testmode<=16))
 	        fprintf_P(stream,diag_portb_P,testmode-9); 
 	      else  
		  	  if ((testmode>=17) && (testmode<=32))
 	          fprintf_P(stream,diag_msel_P,testmode-17); 
      do
      {
      	ch=fgetc(stream);
      } while ((ch!='+') && (ch!='-') && (ch!=27));
      if (ch=='+')
      {
        if (testmode<32) testmode++; else testmode=1;
      }  
      else
        if (ch=='-')
        {
          if (testmode>1) testmode--; else testmode=32;	
        }  
    } while (ch!=27);
    testmode=0;
  }	
  return 0;
}

/*
 * NSLookup
 */
int cmdNSLookup(FILE *stream)
{
	char *hostname;
	u_long ip;
  hostname=strtoken(NULL);
  if (!hostname) return 3;
  ip=NutDnsGetHostByName(hostname);
  if (ip)
    fprintf_P(stream,PSTR("\r\nAdress :%s\r\n"),inet_ntoa(ip));
  else
    fprintf_P(stream,PSTR("\r\nCannot resolve name '%s'\r\n"),hostname);  
  return 0;
}

/*
 * Ping
 */
int cmdPing(FILE *stream)
{
	char *hostname;
  hostname=strtoken(NULL);
  if (!hostname) return 3;
  return 0;
  //return ping(hostname,4,stream);
}

typedef struct
{
  char *cmd;
  int (*func)(FILE *stream);
} TCOMMAND;

TCOMMAND commands[]=
{
  {"PRINT",cmdPrint},
  {"PIXEL" ,cmdPixel},
  {"LINE" ,cmdLine},
  {"HLINE" ,cmdHLine},
  {"VLINE" ,cmdVLine},
  {"CIRCLE", cmdCircle},
  {"FCIRCLE", cmdFCircle},
  {"RECT", cmdRect},
  {"FRECT", cmdFRect},
  {"TEXTCOLOR",cmdTextcolor},
  {"LINECOLOR",cmdLinecolor},
  {"COLOR",cmdColor},
  {"FONT",cmdFont},
  {"DRAWDISPLAY",cmdDraw},
  {"CLS",cmdCls},
  {"HELP",cmdHelp},
  {"INFO",cmdInfo},
  {"SET",cmdSet},
  {"SAVE",cmdSave},
  {"?",cmdHelp},
  {"QUIT",cmdQuit},
  {"POWER",cmdPower},
  {"DIAG",cmdDiag},
  {"PING",cmdPing},
  {"NSLOOKUP",cmdNSLookup},
  {"TEST",cmdTest},
  {NULL ,NULL}
};

/*
 * process a command line
 */
int Processline(char *line, FILE *stream)
{
  short i,err=0;
  char *cmd;
  if (!trimline(line)) return 0;
  cmd=strtoken(line);
  i=0;
  while (commands[i].cmd && stricmp(cmd,commands[i].cmd)) i++;
  if (commands[i].cmd) 
  {
    err=commands[i].func(stream);
    if ((!err) && alwaysdraw) LEDDraw();
  }
  else
    err=6;
  if (err>2)
  	fprintf(stream,"\r\nERROR: %s\r\n",line); 
  switch(err)
  {
    case 3: fputs("ERROR: Missing parameter\r\n",stream); break;
    case 4: fputs("ERROR: Wrong parameter\r\n",stream); break;
    case 5: fputs("ERROR: Cannot load font\r\n",stream); break;
	  case 6: fputs("ERROR: Unknown command\r\n",stream); break;
  }
  return err;
}

