/* LED Matrix display driver (LED panel drawing routines)
 *
 * Controlling several LED Matrix Modules Samsung SLM1606M/SLM1608M
 * connected to port B/D of Ethernut:
 *
 * Printer   Module
 * -----------------
 * D0 (2)    SELECT (CN2-2)
 * D1 (3)    RED    (CN3-2)
 * D2 (4)    GREEN  (CN3-4)
 * D3 (6)    CLOCK  (CN3-6)
 * D4 (8)    BRIGHT (CN3-8)
 * D5 (9)    RESET  (CN3-10)
 * GND (20)  GND    (CN3-3)
 *
 * Written by Thorsten Erdmann 06/2003 (thorsten.erdmann@gmx.de)
 */

/**************************************************************
 **** CAUTION: The module can be destroyed if applied      ****
 ****          static signals, so NEVER interrupt the      ****
 ****          program without switching off module's      ****
 ****          power source first !!!                      ****
 **************************************************************/
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <io.h>

#include <sys/heap.h>
#include <sys/timer.h>
#include <sys/thread.h>

#include "led.h"
#include "testbild.c"


#define MSELECT 0x10
#define MCLOCK  0x20
#define MBRIGHT 0x40
#define MRESET  0x80
#define LEDTIMEOUT 120

#include "chrsmall.c"

static char altered;
static char *bufgreen;
static char *bufred;
static int ledtimeout;
char testmode;

int    clipx0,clipy0,clipx1,clipy1;
PGM_P  font;

/*
 * Startup Timer2 ISR for muxing the panel
 */
void startTimer(void)
{
#ifdef ETHERNUT2	
  /* set timer 2 to CTC mode, clock / 256 = 16uS */
  TCCR2 = BV(CTC2) | BV(CS22);
  /* Set output compare register to 10 = 160uS for now */
  OCR2 = 10;
#else
  /* set timer 2 to CTC mode, clock / 256 = 16uS */
  TCCR2 = BV(WGM21) | BV(CS22);
  /* Set output compare register to 10 = 160uS for now */
  OCR2 = 35;
#endif  
  /* enable the output compare match interrupt */
  TIMSK |= BV(OCIE2);
}


/*
 * enable main power supply
 */
void LEDPowerOn(void)
{
	ledtimeout=0;
	/* if already on simply leave */
	if (PORTE&4) return;
  PORTE=PORTE|(4+8);
  DDRE=0x0c;
  NutSleep(1000);
	PORTE=PORTE&~8;
  /* startup multiplexing */
  startTimer();
  /* Redraw last screen */
  LEDDraw();
}
	
/*
 * disable main power supply
 */	
void LEDPowerOff(void)
{
  /* disable multiplexing */
  TIMSK &= ~BV(OCIE2);
  DDRE=0x0c;
  PORTE=PORTE|8;
  NutSleep(500);
  PORTE=PORTE&~4;
}

void clrbit(short b)
{
  outp(inp(PORTD)|b,PORTD);
}

void setbit(short b)
{
  outp(inp(PORTD)&~b,PORTD);
}

void ModuleSelect(short module)
{
  if (module==0xff)
    outp(inp(PORTD)|MSELECT,PORTD);
  else
    outp(((inp(PORTD)&0xf0)|module)&~MSELECT,PORTD);
}


void pulse(short bit)
{
  setbit(bit);
  clrbit(bit);
}

/*
 * Hardware test
 */
void hardwareTest(void)
{
	static unsigned char b;
	if ((testmode>=1) && (testmode<=8))
	  PORTD=PORTD^(1<<(testmode-1));    // pulse PORTD Pin
	else
	  if ((testmode>=9) && (testmode<=16))
	    PORTB=PORTB^(1<<(testmode-9));  // pulse PORTB Pin
	  else
	    if ((testmode>=17) && (testmode<=32)) // pulse Module Select Pin
	    {
	      if (b==0xff)
	        ModuleSelect(b=testmode-17);
	      else
	        ModuleSelect(b=0xff);     
	    }    
}	        

/*
 * Timer2 ISR
 * Multiplex Module, must be called periodically to avoid module destruction 
 */
SIGNAL(SIG_OUTPUT_COMPARE2)
{
  static short x=0,y=0,p=0,m=0xff;
  static short mask=1,bits;

  if (testmode) hardwareTest();
  else
  {
  	  
    if (!y)
    {
      pulse(MRESET);
      if (altered && (m==0xff)) 
      {
      	altered++;
      	m=0;
        p=0;
      	mask=1;
      }
    }
    	
    if (m!=0xff)
    {
      //setbit(MBRIGHT);
      ModuleSelect(m);
      for (x=0;x<16;x++)
      {
/*     	
      	bits=0xff;
        if (bufgreen[p]&mask)
          bits&=0xef; //0x10;
        if (bufred[p]&mask)
          bits&=0xfe; //0x01;
        if (bufgreen[p+512]&mask)
          bits&=0xdf; //0x20;
        if (bufred[p+512]&mask)
          bits&=0xfd; //0x02;
        if (bufgreen[p+1024]&mask)
          bits&=0xbf; //0x40;
        if (bufred[p+1024]&mask)
          bits&=0xfb; //0x04;
        if (bufgreen[p+1536]&mask)
          bits&=0x7f; //0x80;
        if (bufred[p+1536]&mask)
          bits&=0xf7; //0x08;
*/          
      	bits=0;
        if (bufgreen[p]&mask)
          bits|=0x10;
        if (bufred[p]&mask)
          bits|=0x01;
        if (bufgreen[p+512]&mask)
          bits|=0x20;
        if (bufred[p+512]&mask)
          bits|=0x02;
        if (bufgreen[p+1024]&mask)
          bits|=0x40;
        if (bufred[p+1024]&mask)
          bits|=0x04;
        if (bufgreen[p+1536]&mask)
          bits|=0x80;
        if (bufred[p+1536]&mask)
          bits|=0x08;

        outp(bits,PORTB);  
        pulse(MCLOCK);
        p++;
      }
      clrbit(MBRIGHT);
    	p-=16;
      mask<<=1;
      y++;
      if (y==8)
      {
    	  mask=1;
       	p+=256;
      }
      if (y==16)
      {
        y=0;
        p-=256;
        p+=16;
        mask=1;
        m++;
        if (m>15) 
        {
          if (altered==2) altered=0;
      	  m=0xff;
          ModuleSelect(m);
        }
      }
    }
    else
    {
    	for (x=0;x<16;x++) pulse(MCLOCK);
  	  y++;
  	  if (y==16) y=0;
    }
  }
}

THREAD(PwrTimer,arg)
{
	for (;;)
	{
 	  /* count up the timeout */
 	  if (ledtimeout<LEDTIMEOUT)
 	  {
      ledtimeout++;
 	    if (ledtimeout==LEDTIMEOUT)
        LEDPowerOff();
    }
    NutSleep(1000);	
  }
}


/*
 * Initialize LED Matrix
 */
void LEDInit(void)
{
  /* Port D is 0xff */
  PORTD=0xff;
  /* Port D is output. */
  DDRD=0xff;
  /* Port B is 0xff */
  PORTB=0xff;
  /* Port B is output. */
  DDRB=0xff;
  
  LEDPowerOn();
  NutThreadCreate("PwrTimer", PwrTimer, 0, 100);
  
 
//  setbit(MRESET);        /* reset module      */
//  setbit(MBRIGHT);       /* switch off module */
//  setbit(MSELECT);

  /* allocate frame buffer */
  bufred=(char *)malloc(4096);
  if (!bufred)
  {
  	puts("Error allocation display memory, System haltet!");
  	for (;;);
  }
  bufgreen=bufred+2048;
  font=chrsmall; /* assign font */
  clipx0=0; clipx1=MAXX;
  clipy0=0; clipy1=MAXY;  
  //startTimer();
  memcpy_P(bufgreen,buffer,2048);
  memcpy_P(bufred,buffer,2048);
	altered=1;
}

char *LEDGetbuffer(void)
{
	return bufred;
}

/*
 * HERE COMES THE DRAWING PRIMITIVES
 */

/*
 * Actualize Panel
 */
void LEDDraw(void)
{
	LEDPowerOn();
	altered=1;
}

/* 
 * Clear Screen
 */
void LEDClear(void)
{
  memset(bufred,0,4096);
}

/*
 * Draw Pixel
 */
void LEDPixel(int x, int y, u_char c)
{
  u_int s;
  if ((x<0)||(x>=MAXX)||(y<0)||(y>=MAXY)) return;
  s=x+(y>>3)*MAXX;
  if (c&1)
    bufred[s]|=  0x01<<(y&0x07);
  else
    bufred[s]&=~(0x01<<(y&0x07));
  if (c&2)
    bufgreen[s]|=  0x01<<(y&0x07);
  else
    bufgreen[s]&=~(0x01<<(y&0x07));
}

/*
 * Draw horizontal line
 */
void LEDHLine(int x, int y, u_int l, u_char c)
{
  u_int s,i;
  u_char w,b;

  if ((x>=MAXX)||(y<0)||(y>=MAXY)) return;
  b=MAXY>>3;
  if (x<0)
  {
    l+=x;
    x=0;
  }
  if (x+l>MAXX)
    l=MAXX-x;
  s=x+(y>>3)*MAXX;
  if (c&1)
  {
    w=0x01<<(y&0x07);
    for (i=0;i<l;i++)
    {
      bufred[s]|=w;
      s++;
    }
  }
  else
  {
    w=~(0x01<<(y&0x07));
    for (i=0;i<l;i++)
    {
      bufred[s]&=w;
      s++;
    }
  }
  s-=l;
  if (c&2)
  {
    w=0x01<<(y&0x07);
    for (i=0;i<l;i++)
    {
      bufgreen[s]|=w;
      s++;
    }
  }
  else
  {
    w=~(0x01<<(y&0x07));
    for (i=0;i<l;i++)
    {
      bufgreen[s]&=w;
      s++;
    }
  }
}

/* 
 * Draw vertical line
 */
void LEDVLine(int x, int y, u_int l, u_char c)
{
  u_int i;
  if ((x>=MAXX)||(x<0)||(y>=MAXY)) return;
  for (i=0;i<l;i++) LEDPixel(x,y+i,c);
}

/*
 * Draw Rectangle
 */
void LEDRectangle(int x, int y, int w, int h, u_char c)
{
  LEDHLine(x,y,w,c);
  LEDHLine(x,y+h-1,w,c);
  LEDVLine(x,y,h,c);
  LEDVLine(x+w-1,y,h,c);
}

/*
 * Draw filled Rectangle
 */
void LEDFRectangle(int x, int y, int w, int h, u_char c)
{
  int i;
  for (i=0;i<h;i++)
    LEDHLine(x,y+i,w,c);
}

/* 
 * Draw character
 */
int LEDChar(int x, int y, u_char c, u_char ch)
{
  int adr,d,n,x0,x1,y0,y1,k,w,bpr,bpc,s,b;

  if ((ch<PRG_RDB(font)) || (ch>PRG_RDB(font+1))) return x;
  b=MAXY>>3;
  n=ch-PRG_RDB(font);
  /* get character width */
  w=PRG_RDB(font+3+3*n);
  /* get offset to character bitmap */
  k=PRG_RDB(font+3+3*n+1)+(PRG_RDB(font+3+3*n+2)<<8);
  /* calculate byte per character column */
  bpc=(PRG_RDB(font+2)+7)/8;
  bpr=bpc;
  /* calculate y byte adress */
  y0=y>>3;
  /* clip x coordinates */
  if (x>=clipx0)
    x0=x;
  else
  {
    if (x+w<=clipx0) return x+w;
    x0=clipx0;
  }
  /* calculate bit shift of character row */
  s=y&7;
  
  if (y0>b)
  {
    bpr-=y0;
    y0=b;
  }
    
  /* calculate start adress */
  adr=x0+y0*MAXX;
  n=k+bpc-1+(x0-x)*bpc;
  y1=0;
  while ((y0<b) && (y1<=bpr))
  {
    k=n;
    d=adr;
    for (x1=x0;(x1<x+w)&&(x1<clipx1);x1++)
    {
    	if (c&1)
    	{
        if (y1<bpc) bufred[d]|=PRG_RDB(font+3+k)<<s;
        if (y1>0)   bufred[d]|=(PRG_RDB(font+3+k+1)>>(8-s));
      }
    	if (c&2)
    	{
        if (y1<bpc) bufgreen[d]|=PRG_RDB(font+3+k)<<s;
        if (y1>0)   bufgreen[d]|=(PRG_RDB(font+3+k+1)>>(8-s));
      }
      d++;
      k+=bpc;
    }
    adr+=MAXX;
    n--;
    y0++;
    y1++;
  }
  return x+w;
}

/*
 * calculate character width
 */
int LEDGetChrLength(u_char ch)
{
  if ((ch<PRG_RDB(font)) || (ch>PRG_RDB(font+1))) return 1;
  /* get character width */
  return PRG_RDB(font+3+3*(ch-PRG_RDB(font)));
}

/*
 * calculate character width
 */
int LEDGetStrHeight(void)
{
  /* get character height */
  return PRG_RDB(font+2);
}

/*
 * calculate string length
 */
int LEDGetStrLength(const char *s)
{
  int i=0, l=0;
  while (s[i]!=0)
    l+=LEDGetChrLength(s[i++]);
  return l;
}

/*
 * Draw string
 */
int LEDStr(int x, int y, u_char c, const char *s)
{
  int i;

  LEDFRectangle(x,y,LEDGetStrLength(s),PRG_RDB(font+2),0);
  i=0;
  while ((s[i]!=0) && (x<MAXX))
    x=LEDChar(x,y,c,s[i++]);
  return x;
}

/*
 * Draw formatted string
 */
void LEDFStr(int x, int y, int w, int f, u_char c, const char *s)
{
  int i,x0,l;

  l=LEDGetStrLength(s);
  if (x+w>MAXX) w=MAXX-x;
  switch (f)
  {
    case TE_CENTER: x0=x+((w-l)>>1); break;
    case TE_RIGHT : x0=x+w-l; break;
    default       : x0=x; break;
  }
  if (x>0) clipx0=x;
  if (x+w<MAXX) clipx1=x+w;
  LEDFRectangle(clipx0,y,clipx1-clipx0,PRG_RDB(font+2),0);  // Clear text area
  i=0;
  while ((s[i]!=0) && (x<MAXX))
  {
    x0=LEDChar(x0,y,c,s[i++]);
  }
  clipx0=0;
  clipx1=MAXX;
}

/* 
 * Draw a Circle
 */
void LEDCircle(int x,int y,int r,int c)
{
  int d,xx,yy;
  if (r<2) return;
  d=3-2*r;
  xx=0;
  yy=r;
  while(xx<=yy)
  {
    LEDPixel(x+xx,y+yy,c);
    LEDPixel(x+xx,y-yy,c);
    LEDPixel(x-xx,y+yy,c);
    LEDPixel(x-xx,y-yy,c);
    LEDPixel(x+yy,y+xx,c);
    LEDPixel(x+yy,y-xx,c);
    LEDPixel(x-yy,y+xx,c);
    LEDPixel(x-yy,y-xx,c);
    xx++;
    if (d<0)
    {
      d=d+4*xx+2;
    }
    else
    {
      d=d+4*xx+2-4*(yy-1);
      yy=yy-1;
    }
  }
}

/* 
 * Draw a filled Circle 
 */
void LEDFCircle(int x,int y,int r,int c)
{
  int d,xx,yy;
  if (r<2) return;
  d=3-2*r;
  xx=0;
  yy=r;
  while(xx<=yy)
  {
    LEDHLine(x-xx,y+yy,2*xx,c);
    LEDHLine(x-xx,y-yy,2*xx,c);
    LEDHLine(x-yy,y-xx,2*yy,c);
    LEDHLine(x-yy,y+xx,2*yy,c);
    xx++;
    if (d<0)
    {
      d=d+4*xx+2;
    }
    else
    {
      d=d+4*xx+2-4*(yy-1);
      yy=yy-1;
    }
  }
}

/*
 * Draw Line 
 */
void LEDLine(int x1,int y1,int x2,int y2, int c)
{
  int a,x,y,xInc,yInc;
  int m,d,dx,dy;
  x=x1;
  y=y1;
  d=0;
  dx=x2-x1;
  dy=y2-y1;
  xInc=1;
  yInc=1;
  if (dx<0)
  {
    xInc=-1;
    dx=-dx;
  }
  if (dy<0)
  {
    yInc=-1;
    dy=-dy;
  }
  if (dy<=dx)
  {
    a=2*dx;
    m=2*dy;

    for (x=x1;x<=x2;x+=xInc)
    {
      LEDPixel(x,y,c);
      d+=m;
      if (d>dx)
      {
        y+=yInc;
        d-=a;
      }
    }
  }
  else
  {
    a=2*dy;
    m=2*dx;

    for (y=y1;y<=y2;y+=yInc)
    {
      LEDPixel(x,y,c);
      d+=m;
      if (d>dy)
      {
        x+=xInc;
        d-=a;
      }
    }
  }
}
